/*
 *
 *  Copyright (C) 2008-2009 RICOH Co.,LTD.
 *  All rights reserved.
 *
 *  affiliation	:DSS Development Center
 *  		 Document Solutions & Services Division
 * 
 *  purpose	: OCF CardServiceFactory sample.
 *
 */

package ocfbundle;

import java.util.Enumeration;
import java.util.Vector;

import opencard.core.service.CardChannel;
import opencard.core.service.CardServiceFactory;
import opencard.core.service.CardServiceScheduler;
import opencard.core.service.CardType;
import opencard.core.terminal.CardID;
import opencard.core.terminal.CardTerminalException;
import opencard.core.terminal.CommandAPDU;
import opencard.core.terminal.ResponseAPDU;

/**
 * MyFactory class
 * This class extends CardServiceFactory.
 * This is a card service factory of MyAppletService and MyFileAccessService.
 * 
 * MyFactoryクラス
 * CardserviceFactoryを継承しています。
 * MyAppletServiceとMyFileAccessServiceの、カードサービスファクトリです。
 */
public class MyFactory extends CardServiceFactory {
	
	/**
	 * Card types which are supported by this CardServiceFactory.
	 * 
	 * このカードサービスファクトリがサポートするカードのカードタイプ。
	 */
	private static final int TYPE_CRYPTO   = 0x10;
	private static final int TYPE_JAVAOP = 0x20;
	
	/**
	 * Card ATRs which are supported by this CardServiceFactory.
	 * 
	 * このカードサービスファクトリがサポートするカードのATR。
	 */
	private static final byte[][] ATR_CRYPTO = {
		{(byte)0x3B,(byte)0x95,(byte)0x00,(byte)0x40,(byte)0xFF,(byte)0x63,(byte)0x01,(byte)0x01,(byte)0x00,(byte)0x00},
		{(byte)0x3B,(byte)0x95,(byte)0x00,(byte)0x40,(byte)0xFF,(byte)0x64,(byte)0x02,(byte)0x01,(byte)0x00,(byte)0x00},
		{(byte)0x3B,(byte)0x95,(byte)0x00,(byte)0x40,(byte)0xFF,(byte)0x62,(byte)0x04,(byte)0x01,(byte)0x00,(byte)0x00},
		{(byte)0x3B,(byte)0xE2,(byte)0x00,(byte)0x00,(byte)0x40,(byte)0x20,(byte)0x49,(byte)0x00},
		{(byte)0x3B,(byte)0x85,(byte)0x40,(byte)0x20,(byte)0x68,(byte)0x01,(byte)0x01,(byte)0x00,(byte)0x00},
		{(byte)0x3B,(byte)0x95,(byte)0x15,(byte)0x40,(byte)0x00,(byte)0x68,(byte)0x01,(byte)0x02,(byte)0x00,(byte)0x00},
		{(byte)0x3B,(byte)0x95,(byte)0x00,(byte)0x40,(byte)0xFF,(byte)0x62,(byte)0x01,(byte)0x01,(byte)0x00,(byte)0x00},
		{(byte)0x3B,(byte)0x95,(byte)0x00,(byte)0x40,(byte)0xFF,(byte)0x62,(byte)0x01,(byte)0x02,(byte)0x00,(byte)0x00},
	};
	
	/**
	 * Card ATR masks which are supported by this CardServiceFactory.
	 * 
	 * このカードサービスファクトリがサポートするカードのビットマスク。
	 */
	private static final byte[][] MASK_CRYPTO = {
		{(byte)0xff,(byte)0xff,(byte)0x00,(byte)0xff,(byte)0xff,(byte)0xff,(byte)0xff,(byte)0xff,(byte)0x00,(byte)0x00},
		{(byte)0xff,(byte)0xff,(byte)0x00,(byte)0xff,(byte)0xff,(byte)0xff,(byte)0xff,(byte)0xff,(byte)0x00,(byte)0x00},
		{(byte)0xff,(byte)0xff,(byte)0x00,(byte)0xff,(byte)0xff,(byte)0xff,(byte)0xff,(byte)0xff,(byte)0x00,(byte)0x00},
		{(byte)0xff,(byte)0xff,(byte)0x00,(byte)0x00,(byte)0xff,(byte)0xff,(byte)0xff,(byte)0x00},
		{(byte)0xff,(byte)0xff,(byte)0xff,(byte)0xff,(byte)0xff,(byte)0xff,(byte)0xff,(byte)0x00,(byte)0x00},
		{(byte)0xff,(byte)0xff,(byte)0xff,(byte)0xff,(byte)0x00,(byte)0xff,(byte)0xff,(byte)0xff,(byte)0x00,(byte)0x00},
		{(byte)0xff,(byte)0xff,(byte)0x00,(byte)0xff,(byte)0xff,(byte)0xff,(byte)0xff,(byte)0xff,(byte)0x00,(byte)0x00},
		{(byte)0xff,(byte)0xff,(byte)0x00,(byte)0xff,(byte)0xff,(byte)0xff,(byte)0xff,(byte)0xff,(byte)0x00,(byte)0x00},
	};
	
	/**
	 * The AtrFilter object to be used to know if the card type is TYPE_CRYPTO.
	 * 
	 * カードタイプがTYPE_CRYPTOかを判別するために使用するATRフィルタ。
	 */
	private static AtrFilter fileAtrFilter;
	/*
	 * initialization of the fileAtrFilter.
	 * fileAtrFilter初期化
	 */
	static {
		fileAtrFilter = new AtrFilter();
		for (int i = 0; i < ATR_CRYPTO.length; i++) {
			fileAtrFilter.put(MASK_CRYPTO[i], ATR_CRYPTO[i]);	
		}
	}
	
	/**
	 * Command bytes to be used to know if the card type is TYPE_JAVAOP.
	 * 
	 * カードタイプがTYPE_JAVAOPかを判別するためにカードに送信するコマンド。
	 */
	private static final byte[] GETDATA_CPLC = new byte[]{
		(byte)0x80, (byte)0xCA, (byte)0x9F, (byte)0x7F, (byte)0x2D
	};
	/**
	 * Expected response code of a TYPE_JAVAOP card.
	 * 
	 * カードタイプがTYPE_JAVAOPであった場合、期待されるレスポンスコード。
	 */
	private static final byte SW1_OK = (byte)0x90;
	
	/**
	 * Vectors which stores card service classes provided by this CardServiceFactory.
	 * 
	 * このカードサービスファクトリが提供するカードサービスを格納するVector。
	 */
	private static Vector fileServices;
	private static Vector appletServices;
	/*
	 * initialization of the card service vectors.
	 * カードサービス初期化
	 */
	static {
		Vector fileVec = new Vector();
		fileVec.add(MyFileAccessService.class);
		fileServices = fileVec;
		
		Vector appVec = new Vector();
		appVec.add(MyAppletService.class);
		appletServices = appVec;
	}
	
	/* (non-Javadoc)
	 * @see opencard.core.service.CardServiceFactory#getCardType(opencard.core.terminal.CardID, opencard.core.service.CardServiceScheduler)
	 */
	protected CardType getCardType(CardID cardID, CardServiceScheduler scheduler) throws CardTerminalException {
		/*
		 * returns a card type corresponding to the given card
		 * カードタイプを判別する
		 */
		
		if (fileAtrFilter.isCandidate(cardID)) {
			CardType type = new CardType(TYPE_CRYPTO);
			return type;
			
		}
		
		CardChannel channel = null;
		try {
			channel = scheduler.allocateCardChannel(this, true);
			ResponseAPDU response = 
				channel.sendCommandAPDU(new CommandAPDU(GETDATA_CPLC));
			if (response.sw1() == SW1_OK) {
				CardType type = new CardType(TYPE_JAVAOP);
				return type;
			}
		} finally {
			if (channel != null) scheduler.releaseCardChannel(channel);
		}
		
		return CardType.UNSUPPORTED;
	}
	
	/* (non-Javadoc)
	 * @see opencard.core.service.CardServiceFactory#getClasses(opencard.core.service.CardType)
	 */
	protected Enumeration getClasses(CardType cardType) {
		/*
		 * returns card services corresponding to the given card type
		 * カードタイプに応じて、カードサービスを返す
		 */
		if (TYPE_CRYPTO == cardType.getType()) {
			return fileServices.elements();
			
		} else if (TYPE_JAVAOP == cardType.getType()) {
			return appletServices.elements();
			
		} else {
			return null;
		}
	}
}
